<?php
/**
 * Class serving user application requests
 */
class Request extends Standard {
	// error codes
	const ERROR_UNKNOWN = 0;
	const ERROR_BAD_REQUEST = 1;
	const ERROR_REQUIRED_FIELD_MISSING = 2;
	const ERROR_FIELD_MALFORMED = 3;
	const ERROR_DUPLICATE_ENTRY = 4;
	const ERROR_NOT_FOUND = 5;
	const ERROR_NOT_ALLOWED = 6;
	const ERROR_UPLOAD = 100; // usage: self::ERROR_UPLOAD + UPLOAD_ERR_PARTIAL
	const ANONYMOUS_PREFIX = 'anonym';
	protected $action;
	protected $args;
	protected $session;
	protected $user;
	protected $files;

	public function __construct($request_string){
		parent::__construct();
    $this->parse($request_string);
    
    // do clean-up
    if (empty($_SESSION['cleaned_up']) OR ($_SESSION['cleaned_up'] < time() - SESSION_TIMEOUT)) {
		  $this->cleanup_all();
      $_SESSION['cleaned_up'] = time();
		}
	  
	  // new session
	  if(!isset($_SESSION['request']['id'])){
	    $_SESSION['request'] = array();
	    $this->user = new LoggedUser();
	    
	  // restore old session
	  } else {
	    $this->user = new LoggedUser($_SESSION['request']['id']);
      $this->user->update_session();
	  }
	  
	  $this->session = &$_SESSION['request'];
	  
	  if(!empty($_FILES)) $this->files = $_FILES;
	}

	protected function parse($request_string){
		$request = json_decode($request_string, true);
		if(empty($request) OR empty($request['action']) OR !isset($request['args'])) return false;

		$this->action = $request['action'];
		$this->args = $request['args'];
	}

	public function serve(){
		$result = null;
		if(method_exists($this, 'serve_' . $this->action)){
			try {
				$result = call_user_func_array(array($this, 'serve_' . $this->action), array($this->args));
			} catch (Exception $e) {
				$result = $e->toArray();
			}
		} else {
			$result = null;
		}
		return json_encode($result);
	}

	/**
	 * Registers new user
	 * - checks passed arguments - especially whether username is unique
	 * posible return values:
	 *   - error - missing values or unicity of user
	 *   - boolean true - user was successfuly registered
	 */
	protected function serve_registration($args){
		// check if fields are present
		if(empty($args['login']) OR empty($args['email']) OR empty($args['pass'])){
			throw new RequestException("Nebolo vyplnené niektoré z políčok prihlasovacie meno, e-mail, heslo.", self::ERROR_REQUIRED_FIELD_MISSING);
		}

		// check formats
		if(false === ereg('^[a-zA-Z0-9\_]+$', $args['login'])){
			throw new RequestException("Zadané prihlasovacie meno je v nesprávnom formáte.",  self::ERROR_FIELD_MALFORMED);
		}

		if(false === ereg(EMAIL_REGEX, $args['email'])){
			throw new RequestException("Zadaný e-mail je v nesprávnom formáte.",  self::ERROR_FIELD_MALFORMED);
		}

		$args['pass'] = sha1($args['pass']);
		$args['login'] = sha1($args['login']);
		
    // create registration key
    $regkey = mt_rand(10000, 9999999999) . sha1($args['login']) . mt_rand(10000, 9999999999) . sha1($args['email']) . mt_rand(10000, 9999999999);
    $regkey = sha1($regkey);
    
		// save registration to db
		if(!$this->db->insert(array(), 'registrations', array('columns' => array('nick', 'login', 'pass', 'email', 'time', 'regkey'), 'values' => array(array($args['nick'], $args['login'], $args['pass'], $args['email'], '{NOW()}', $regkey))))){
			if($this->db->getLastErrorCode() == 23000){
				throw new RequestException("Používateľ s takýmto používateľským menom už existuje.", self::ERROR_DUPLICATE_ENTRY);
			} else {
				throw new RequestException("Registráciu sa nepodarilo uložiť do databázy.", self::ERROR_UNKNOWN);
			}
		}
		
		/* send registration confirmation email */
		$link = DOMAIN . "registration_confirmation.php?code=$regkey";

		$subject = "CLIM registration confirmation";
		$body = "You requested a registration to CLIM. <br> You supported: <br>
		  <blockqoute>
		    login: '{$args['nick']}'<br>
		    email: '{$args['email']}'<br> 
	    </blockquote>
	    Please click folowing link to confirm your registration: $link";
	  
		return @mail_it($args['email'], $subject, $body);
	}

	/**
	 * Checks registration code and acitavets user registration
	 */
  protected function serve_registration_confirmation($args){
  	$reg = $this->db->select(array(), array("registrations"), "*", array('goperator' => 'and', array('id_user', '=', '0'), array('regkey', '=', "'" . $this->db->sanitize_input($args['code']) . "'")));
  	if(!empty($reg)) $reg = current($reg);
    
    // registration code not found
    if(empty($reg)) return "Registration failed - wrong code.";
    
    // create new user
    $uid = $this->db->insert(array(), 'users', 
      array('columns' => array('nick', 'login', 'pass', 'email', 'active'), 
        'values' => array(array($this->db->sanitize_input($reg['nick']), $reg['login'], $reg['pass'], $this->db->sanitize_input($reg['email']), 1))
      )
    );
    
    if(!$uid){
      return 'Failed to create user.';
    }

    $this->db->update(array(), 'registrations', array('id_user' => $uid), array('goperator' => 'and', array('id', '=', $reg['id'])));
    
    return 'Registration completed successfully.';
  }
  
  /**
   * Registers new anonymous user
   * @returns user information hashes (username and password)
   */
  protected function serve_registration_anonymous($args){
    // find unused anonymous username
    do{
      // TODO - vacsi rozsah - pridanim pismen
      $nick = self::ANONYMOUS_PREFIX . mt_rand(1,999);
      $login = sha1($nick);
      
      $_tmp = $this->db->select(array(), array('users'), '*', array('login', '=', "'".sha1($login)."'"));
    } while(!empty($_tmp));
    
    $pass = sha1(mt_rand(10000, 9999999999) . $login . mt_rand(10000, 9999999999) . mt_rand(10000, 9999999999));

    // save registration to db
    if(!$this->db->insert(array(), 'users', array('columns' => array('nick', 'login', 'pass', 'anonymous'), 'values' => array(array($nick, sha1($login), sha1($pass), 1))))){
      if($this->db->getLastErrorCode() == 23000){
        return $this->serve_registration_anonymous($args);
        
      } else {
        throw new RequestException("Používateľa sa nepodarilo uložiť do databázy.", self::ERROR_UNKNOWN);
      }
    }
    
    return array('login' => $login, 'pass' => $pass);
  }
  
  /**
   * Checks supported login information and saves session
   * @return user information
   */
  protected function serve_login($args){
    if($this->user->is_logged_in()) return $this->user->get_info();
    
    if(empty($args['login']) OR empty($args['pass'])) throw new RequestException("Nebolo vyplnené niektoré z políčok používateľské meno, heslo.", self::ERROR_REQUIRED_FIELD_MISSING);
    
    // try to log user in
    $this->user = new LoggedUser(null, sha1($args['login']), sha1($args['pass']));
    
    // check if login was successful
    if(!$this->user->is_correct()) throw new RequestException("Používateľ s takýmto prihlasovacím menom neexistuje.", self::ERROR_NOT_FOUND);
        
    $this->user->update_session();
    $this->session['id'] = $this->user->get_id();
    
    return array('user' => $this->user->get_info(), 'contacts' => $this->user->get_contacts());
  }
  
  /**
   * Logs current user out - sets session logout time
   */
  protected function serve_logout($args){
  	$this->session = array();
    return $this->user->logout();
  }
  
  protected function serve_remove_contact($args){
    if(!$this->user->is_logged_in()) throw new RequestException("Na vykonanie tejto akcie musíte byť prihlásený.", self::ERROR_BAD_REQUEST);
    if(empty($args['id_contact'])) throw new RequestException("Nebol vybraný kontakt.", self::ERROR_REQUIRED_FIELD_MISSING);
    $this->user->remove_contact((int) $args['id_contact']);
    
    return array('contacts' => $this->user->get_contacts());
  }
  
  protected function serve_add_contact($args){
    if(!$this->user->is_logged_in()) throw new RequestException("Na vykonanie tejto akcie musíte byť prihlásený.", self::ERROR_BAD_REQUEST);
    if(empty($args['id_contact'])) throw new RequestException("Nebol vybraný kontakt.", self::ERROR_REQUIRED_FIELD_MISSING);
    $this->user->add_contact((int) $args['id_contact']);
    
    return array('contacts' => $this->user->get_contacts());    
  }

  protected function serve_search_contacts($args){
    if(!$this->user->is_logged_in()) throw new RequestException("Na vykonanie tejto akcie musíte byť prihlásený.", self::ERROR_BAD_REQUEST);
    if(empty($args['search'])) throw new RequestException("Nebol zadaný vyhľadávací reťazec.", self::ERROR_REQUIRED_FIELD_MISSING);
    
    $args['search'] = $this->db->sanitize_input($args['search']);
    $args['search'] = trim($args['search'], "'");
    $list = $this->db->select(
      array(), array("users"), '*', 
      array(
        'goperator' => 'and', 
        array(
          'goperator' => 'or', 
          array("nick", "LIKE", "'%{$args['search']}%'"), 
          array("email", "LIKE", "'%{$args['search']}%@%'")
        ),
        array(
          "id", '!=', $this->user->get_id()
        ),
        array(
          $this->db->prepare_set_stmt(array_keys($this->user->get_contacts()), 'id', false)
        )
      ),
      array('count' => 10, 'offset' => 0)
    );
    
    if(empty($list)){
      $contacts = array();
      //throw new RequestException("Na základe zadaných kritérií sa nenašli žiadne kontakty.", self::ERROR_NOT_FOUND);
    } else {
      $contacts = array();
      foreach($list as $row){
        $_contact = $this->contacts[$row['id']] = new Contact($row['id']);
        $contacts[] = $_contact->get_info();
      }
    }
      
    return array('contacts' => $contacts);
  }
  
  protected function serve_send_message($args){
    if(!$this->user->is_logged_in()) throw new RequestException("Na vykonanie tejto akcie musíte byť prihlásený.", self::ERROR_BAD_REQUEST);
    if(empty($args['id_user_contact']) OR ((int) $args['id_user_contact'] == 0)) throw new RequestException("Nebol vybraný kontakt.", self::ERROR_REQUIRED_FIELD_MISSING);
    if(!isset($args['text'])) throw new RequestException("Nebol zadaný text správy.", self::ERROR_REQUIRED_FIELD_MISSING);
    
    $args['id_user_contact'] = (int) $args['id_user_contact'];

    // check if contact is in users contact list
    if(!$this->user->has_contact($args['id_user_contact'])) throw new RequestException("Príjemca sa nenachádza vo Vašom zozname kontaktov.", self::ERROR_NOT_ALLOWED);
    
    // empty text? do not send anything
    if($args['text'] == '') return;
    $args['text'] = htmlspecialchars($args['text']);
    
    $sent_time = time();
    // save message
    if(!$this->db->insert(array(), 'posts', array('columns' => array('id_user', 'id_user_contact', 'text', 'sent_time'), 'values' => array(array($this->user->get_id(), $args['id_user_contact'], $args['text'], date('Y-m-d H:i:s', $sent_time)))))){
      throw new RequestException("Správu sa nepodarilo odoslať, skúste to prosím znovu.", self::ERROR_UNKNOWN);
    }
    
    return array('time' => $sent_time, 'text' => $args['text']);
  }
  
  protected function serve_send_file($args){
    if(!$this->user->is_logged_in()) throw new RequestException("Na vykonanie tejto akcie musíte byť prihlásený.", self::ERROR_BAD_REQUEST);
    if(empty($args['id_user_contact']) OR ((int) $args['id_user_contact'] == 0)) throw new RequestException("Nebol vybraný kontakt.", self::ERROR_REQUIRED_FIELD_MISSING);
    if(empty($this->files)) throw new RequestException("Nebol vybraný žiadny súbor.", self::ERROR_REQUIRED_FIELD_MISSING);
    
    $args['id_user_contact'] = (int) $args['id_user_contact'];
    
    // check if contact is in users contact list
    if(!$this->user->has_contact($args['id_user_contact'])) throw new RequestException("Príjemca sa nenachádza vo Vašom zozname kontaktov.", self::ERROR_NOT_ALLOWED);
    
    $maxsize = get_shorthand_bytes(UPLOAD_MAX_FILESIZE, 0);
    $file = reset($this->files);
    $filename = basename($file['name']);
    
    // check file for errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
      switch($file['error']){
        case UPLOAD_ERR_INI_SIZE:  
        case UPLOAD_ERR_FORM_SIZE: 
          throw new RequestException("Súbor '$filename' je príliš veľký. Maximálna veľkosť je $maxsize.", self::ERROR_UPLOAD + $file['error']);
        case UPLOAD_ERR_PARTIAL:
          throw new RequestException("Súbor '$filename' nebol nahraný úplne celý.", self::ERROR_UPLOAD + $file['error']); 
        case UPLOAD_ERR_NO_FILE: 
          throw new RequestException("Nebol vybraný žiadny súbor.", self::ERROR_UPLOAD + $file['error']);
        case UPLOAD_ERR_NO_TMP_DIR: 
          throw new RequestException("Server: neexistuje dočasný adresár.", self::ERROR_UPLOAD + $file['error']);
        case UPLOAD_ERR_CANT_WRITE: 
          throw new RequestException("Server: súbor '$filename' sa nepodarilo uložiť.", self::ERROR_UPLOAD + $file['error']);
        case UPLOAD_ERR_EXTENSION: 
          throw new RequestException("Nepovolená prípona súboru '$filename'.", self::ERROR_UPLOAD + $file['error']);
        default: 
          throw new RequestException("Neznáma chyba pri nahrávaní súboru '$filename'.", self::ERROR_UPLOAD + $file['error']);
      }
    }
    
    // Move uploaded files to upload_dir/userID/
    $target_dir = DIRECTORY_UPLOADS . '/' . $this->user->get_id();
    if(!is_dir($target_dir)) {
      if(!mkdir($target_dir)){
        throw new RequestException("Server: nepodarilo sa vytvoriť adresár používateľa.", self::ERROR_UPLOAD);
      } else {
        chmod($target_dir, DIRECTORY_UPLOADS_PERMISSIONS);
      }
    }
    
    $target_path = $target_dir . '/' . (microtime(true) * 100) . mt_rand(10000, 99999);
    
    // try to move file with different approaches
    if (move_uploaded_file($file['tmp_name'], $target_path)) {
    } elseif (rename($file['tmp_name'], $target_path)) {
    } elseif (copy($file['tmp_name'], $target_path)) {
    } elseif (false === @file_put_contents($target_path, @file_get_contents($file['tmp_name']))) {
    } else {
      throw new RequestException("Server: súbor '$filename' sa nepodarilo presunúť do adresára používateľa.", self::ERROR_UPLOAD);
    } 
    
    chmod($target_path, DIRECTORY_UPLOADS_PERMISSIONS);
    
    $sent_time = time();
    
    $insert = array(
      'id_user' => $this->user->get_id(),
      'id_user_contact' => $args['id_user_contact'],
      'sent_time' => date('Y-m-d H:i:s', $sent_time),
      'name' => $filename,
      'path' => $target_path,
      'size' => $file['size']
    );
    
    // save file transfer
    if(!$this->db->insert(array(), 'file_transfers', array('columns' => array_keys($insert), 'values' => array(array_values($insert))))){
      throw new RequestException("Súbor '$filename' sa nepodarilo odoslať, skúste to prosím znovu.", self::ERROR_UNKNOWN);
    }
    
    return array('time' => $sent_time, 'filename' => $filename);
  }
  
  protected function serve_get_user_info($args){
    if(!$this->user->is_logged_in()) throw new RequestException("Na vykonanie tejto akcie musíte byť prihlásený.", self::ERROR_BAD_REQUEST);
    
  	return array('user' => $this->user->get_info());
  }
  
  protected function serve_get_messages($args){
    if(!$this->user->is_logged_in()) throw new RequestException("Na vykonanie tejto akcie musíte byť prihlásený.", self::ERROR_BAD_REQUEST);
  	
    $messages = $this->db->select(
      array('id' => 'id'), 
      array("posts"), 
      array(array('id'), array('id_user', 'posts', 'id_user_contact'), array('text'), array('sent_time')), 
      array('goperator' => 'and', array('id_user_contact', '=', $this->user->get_id()), array('recieved_time', 'IS', 'NULL')),
      array('count' => null, 'offset' => 0),
      array(array('sent_time', 'dir' => 'asc'))
    );
  	
  	if(!is_array($messages)) throw new RequestException("Nepodarilo sa získať zoznam nových správ", self::ERROR_UNKNOWN);
  	
  	if (!empty($messages)) {
  		$this->db->update(array(), 'posts', array('recieved_time' => '{NOW()}'), $this->db->prepare_set_stmt(array_keys($messages), 'id'));
    }
    		
		$_messages = array();
		foreach($messages as $id => $msg){
			$msg['sent_time'] = strtotime($msg['sent_time']); 
			$_messages[$id] = $msg;
    }
		
		return array('messages' => $_messages);
  }
  
  protected function serve_get_transfers($args){
    if(!$this->user->is_logged_in()) throw new RequestException("Na vykonanie tejto akcie musíte byť prihlásený.", self::ERROR_BAD_REQUEST);
  	
    $transfers = $this->db->select(
      array('id' => 'id'), 
      array("file_transfers"), 
      array(array('id'), array('id_user', 'file_transfers', 'id_user_contact'), array('name'), array('size'), array('sent_time')), 
      array('goperator' => 'and', array('id_user_contact', '=', $this->user->get_id()), array('recieved_time', 'IS', 'NULL'), array('refused_time', 'IS', 'NULL')),
      array('count' => null, 'offset' => 0),
      array(array('sent_time', 'dir' => 'asc'))
    );
  	
  	if(!is_array($transfers)) throw new RequestException("Nepodarilo sa získať zoznam súborov čakajúcich na prijatie", self::ERROR_UNKNOWN);
	
  	if (!empty($transfers)) {
		  $this->db->update(array(), 'file_transfers', array('recieved_time' => '{NOW()}'), $this->db->prepare_set_stmt(array_keys($transfers), 'id'));
		}
		
		$result = array();
		foreach($transfers as $id => $file){
			$file['sent_time'] = strtotime($file['sent_time']); 
			$result[$id] = $file;
    }
		
		return array('transfers' => $result);
  }
  
  protected function serve_download_transfer($args){
    if(!$this->user->is_logged_in()) throw new RequestException("Na vykonanie tejto akcie musíte byť prihlásený.", self::ERROR_BAD_REQUEST);
    if(empty($args['id_transfer']) OR ((int) $args['id_transfer'] == 0)) throw new RequestException("Nebol vybraný žiadny súbor.", self::ERROR_REQUIRED_FIELD_MISSING);
  	$args['id_transfer'] = (int) $args['id_transfer'];
  	
  	// get transfer from database
    $transfer = $this->db->select(
      array('id' => 'id'), 
      array("file_transfers"), 
      array(array('id'), array('id_user_contact'), array('name'), array('path'), array('size'), array('sent_time')), 
      array('goperator' => 'and', array('id', '=', $args['id_transfer'])),
      array('count' => null, 'offset' => 0),
      array(array('sent_time', 'dir' => 'asc'))
    );
    
    $transfer = reset($transfer);
    
    // check if transfer was found and belongs to current user
    if(empty($transfer) OR ($transfer['id_user_contact'] != $this->user->get_id())){
      throw new RequestException("Súbor sa nenašiel.", self::ERROR_NOT_FOUND);
    }
  	
  	if(!is_readable($transfer['path'])) throw new RequestException("Server: súbor sa nenašiel.", self::ERROR_REQUIRED_FIELD_MISSING);
  	
  	do {
  	  $key = mt_rand(10000000, 9999999999999);
  	} while(isset($_SESSION['download'][$key]));
  	
  	$_SESSION['download'][$key] = $transfer['path'];
    return array('name' => $transfer['name'], 'size' => $transfer['size'], 'url' => DOMAIN . 'get_file.php?file='.$key);
  }
  
  protected function serve_get_history($args){
    if(!$this->user->is_logged_in()) throw new RequestException("Na vykonanie tejto akcie musíte byť prihlásený.", self::ERROR_BAD_REQUEST);
    if(empty($args['id_user_contact']) OR ((int) $args['id_user_contact'] == 0)) throw new RequestException("Nebol vybraný kontakt.", self::ERROR_REQUIRED_FIELD_MISSING);
  	$args['id_user_contact'] = (int) $args['id_user_contact'];
  	
  	$posts = array();
  	foreach(array('posts_history', 'posts') as $tbl){
	    $posts = array_merge($posts, $this->db->select(
	      array(), 
	      array($tbl), 
	      "*", 
	      array('goperator' => 'or', 
	        array('goperator' => 'and',
	          array('id_user', '=', $args['id_user_contact']),
	          array('id_user_contact', '=', $this->user->get_id())
	        ),
	        array('goperator' => 'and',
	          array('id_user', '=', $this->user->get_id()),
	          array('id_user_contact', '=', $args['id_user_contact'])
	        ),
	      )
	    ));
  	}
  
    // save some band-width - send only needed information	
  	foreach($posts as &$post){
      $post['type'] = ($post['id_user'] == $this->user->get_id()) ? 'sent' : 'recieved';
  		if($post[$post['type'].'_time'] === null) $post[$post['type'].'_time'] = $post['sent_time'];
  		$post['time'] = strtotime($post[$post['type'].'_time']);
  		
  		unset($post['sent_time']);
  		unset($post['recieved_time']);
  		unset($post['id']);
  		unset($post['id_user']);
  		unset($post['id_user_contact']);
  	}
  	
  	usort($posts, array($this, 'sort_posts_aux'));
    return array('messages' => $posts);
  }
  
  protected function serve_get_history_xml($args){
    if(!$this->user->is_logged_in()) throw new RequestException("Na vykonanie tejto akcie musíte byť prihlásený.", self::ERROR_BAD_REQUEST);
    if(empty($args['id_user_contact']) OR ((int) $args['id_user_contact'] == 0)) throw new RequestException("Nebol vybraný kontakt.", self::ERROR_REQUIRED_FIELD_MISSING);
    $args['id_user_contact'] = (int) $args['id_user_contact'];
    
    $posts = array();
    foreach(array('posts_history', 'posts') as $tbl){
      $posts = array_merge($posts, $this->db->select(
        array(), 
        array($tbl), 
        "*", 
        array('goperator' => 'or', 
          array('goperator' => 'and',
            array('id_user', '=', $args['id_user_contact']),
            array('id_user_contact', '=', $this->user->get_id())
          ),
          array('goperator' => 'and',
            array('id_user', '=', $this->user->get_id()),
            array('id_user_contact', '=', $args['id_user_contact'])
          ),
        )
      ));
    }
    
    $contact = $this->user->get_contacts();
    $contact = $contact[$args['id_user_contact']];
    $user = $this->user->get_info();
    
    // save some band-width - send only needed information  
    foreach($posts as &$post){
    	if($post['id_user'] == $this->user->get_id()){
        $post['type'] = 'sent';
        $post['sender'] = $user['nick'];
        $post['recipient'] = $contact['nick'];
    		
    	} else {
    		$post['type'] = 'recieved';
    		$post['sender'] = $contact['nick'];
    		$post['recipient'] = $user['nick'];
    		
    	} 
    	
      if($post[$post['type'].'_time'] === null){
      	$post['time'] = $post['sent_time'];
      } 
      
      if(!isset($post['time'])){
        $post['time'] = strtotime($post[$post['type'].'_time']);
      }
      
      unset($post['id']);
      unset($post['id_user']);
      unset($post['id_user_contact']);
    }
    
    usort($posts, array($this, 'sort_posts_aux'));
    
    $xml = "<?xml version='1.0'?>\r\n";
    $xml .= "<history>\r\n";
    
    foreach($posts as &$post){
      $xml .= "\t<entry>\r\n";
      unset($post['time']);
      foreach($post as $col => $val){
      	$xml .= "\t\t<$col>" . htmlspecialchars($val) . "</$col>\r\n";
      }
      $xml .= "\t</entry>\r\n";
    }
    
    $xml .= "</history>";
    
    return array('xml' => $xml);
  }
  
  /**
   * Auxiliary function for usort and such
   * Sorts by time
   * 
   * @param (array) &$a
   * @param (array) &$b
   * @return int (-1|0|1)
   */
  protected function sort_posts_aux($a, $b){
  	if($a['time'] == $b['time']) return 0;
  	return ($a['time'] < $b['time']) ? -1 : 1;
  }
  
  protected function serve_get_contacts($args){
    if(!$this->user->is_logged_in()) throw new RequestException("Na vykonanie tejto akcie musíte byť prihlásený.", self::ERROR_BAD_REQUEST);
    return array('contacts' => $this->user->get_contacts());
  }
  
  protected function serve_get_information($args){
    if(!$this->user->is_logged_in()) throw new RequestException("Na vykonanie tejto akcie musíte byť prihlásený.", self::ERROR_BAD_REQUEST);
  	if(empty($args)) return array();
  	
  	$result = array();
  	foreach($args as $requested_information => $requested_args){
  		if(!method_exists($this, 'serve_' . $requested_information)){
        throw new RequestException("Požadované neznáme informácie ($requested_information).", self::ERROR_NOT_ALLOWED);
  		}
  		$result = array_merge($result, call_user_func_array(array($this, 'serve_' . $requested_information), array($requested_args)));
  	}
  	return $result;
  }

  protected function serve_set_information($args){
    if(!$this->user->is_logged_in()) throw new RequestException("Na vykonanie tejto akcie musíte byť prihlásený.", self::ERROR_BAD_REQUEST);
  	if(!isset($args['id_user']) OR ((int) $args['id_user'] == 0)) throw new RequestException("Nebol zadaný používateľ.", self::ERROR_REQUIRED_FIELD_MISSING);
  	$args['id_user'] = (int) $args['id_user'];
  	
  	// editacia vlastneho profilu
  	if($this->user->get_id() == $args['id_user']){
      $update = array();
      unset($args['id_user']);
      foreach($args as $name => $value){
      	$update[$name] = $this->db->sanitize_input($value);
      }
      
      return (bool) $this->db->update(
        array(), 
        'users', 
        $update, 
        array('id', '=', $this->user->get_id()),
        1
      );
  		
    // editacia informacii o kontakte
  	} elseif($this->user->has_contact($args['id_user'])) {
      if(!isset($args['nick'])){
      	throw new RequestException("Nebola zadaná nová prezývka.", self::ERROR_REQUIRED_FIELD_MISSING);
      }
      
      return (bool) $this->db->update(
        array(), 
        'contacts', 
        array('alias' => ($args['nick'] == '') ? '{NULL}' : $this->db->sanitize_input($args['nick'])), 
        array('goperator' => 'and', 
          array('id_user', '=', $this->user->get_id()),
          array('id_user_contact', '=', $args['id_user'])
        ),
        1
      );
  		
  	} else {
  		throw new RequestException("Kontakt sa nenachádza vo Vašom zozname.", self::ERROR_NOT_ALLOWED);
  	}
  }
  
  /**
   * Logs inactive sessions out automatically
   * @return void
   */
  protected function cleanup_sessions(){
    $this->db->update(array(), 'sessions', array('logout_time' => '{NOW()}', 'logout_reason' => 'timeout'), array('goperator' => 'and', array('logout_time', 'IS', 'NULL'), array('last_action_time', '<=', "'" . date('Y-m-d H:i:s', time() - SESSION_TIMEOUT) . "'")));
  }

  /**
   * Moves old messages to history
   * @return void
   */
  protected function cleanup_messages(){
  	$msgs = $this->db->select(array('id' => 'id'), array("posts"), "*", array('goperator' => 'and', array('recieved_time', 'IS NOT', 'NULL'), array('recieved_time', '<', 'NOW() - INTERVAL 1 DAY')));
  	if(empty($msgs)) return;
  	
  	$this->db->delete(array(), 'posts', $this->db->prepare_set_stmt(array_keys($msgs), 'id'));
  	
  	$_msg = reset($msgs);
  	$this->db->insert(array(), 'posts_history', array('columns' => array_keys($_msg), 'values' => $msgs));
  }
  
  /**
   * Moves old file transfers to history
   * @return void
   */
  protected function cleanup_transfers(){
  	$transfers = $this->db->select(array('id' => 'id'), array("file_transfers"), "*", array('goperator' => 'and', array('recieved_time', 'IS NOT', 'NULL'), array('recieved_time', '<', 'NOW() - INTERVAL 1 DAY')));
  	if(empty($transfers)) return;
  	
  	$this->db->delete(array(), 'file_transfers', $this->db->prepare_set_stmt(array_keys($transfers), 'id'));
  	
  	$_transfer = reset($transfers);
  	$this->db->insert(array(), 'posts_history', array('columns' => array_keys($_transfer), 'values' => transfers));
  }
  
  /**
   * Removes innactive users
   * @return void
   */
  protected function cleanup_anonymous_users(){
  	// TODO
    //$to_remove = $this->db->select(array('id' => 'id_user'), array("sessions"), array(array('id_user', 'sessions', 'id_user'), array('max(recieved_time)', 'sessions', 'last_time')), null, null, null, array('id_user'));
    
    //if(empty($to_remove)) return;
    
  }
  
  /**
   * Calls all cleanup methods
   * @return void
   */
  protected function cleanup_all(){
  	foreach(get_class_methods(get_class($this)) as $method){
  		if($method == 'cleanup_all') continue;
  		if(strpos($method, 'cleanup_') === 0){
  			call_user_func(array($this, $method));
  		}
  	}
  }
}
?>